package net.mesvaccins.sadv.example;

import com.syadem.sadv.ApiClient;
import com.syadem.sadv.ApiException;
import com.syadem.sadv.DefaultApi;
import com.syadem.sadv.model.DiagnosticRequest;
import com.syadem.sadv.model.ImmunizationAct;
import com.syadem.sadv.model.Patient;
import com.syadem.sadv.model.Patient.GenderEnum;
import java.time.LocalDate;
import java.util.Optional;

public class Main {

  public static void main(String[] args) {
    var client = new ApiClient(
      Optional
        .ofNullable(System.getenv("SADV_CLIENT_ID"))
        .orElseThrow(() ->
          new IllegalStateException(
            "Environment variable SADV_CLIENT_ID is missing."
          )
        ),
      // Client secret
      Optional
        .ofNullable(System.getenv("SADV_CLIENT_SECRET"))
        .orElseThrow(() ->
          new IllegalStateException(
            "Environment variable SADV_CLIENT_SECRET is missing."
          )
        ),
      null
    );
    client.setBasePath("https://api.fr.sad.mesvaccins.net");
    client.setDebugging(true);

    var apiInstance = new DefaultApi(client);
    var diagnosticRequest = new DiagnosticRequest()
      .patient(
        new Patient()
          .gender(GenderEnum.M)
          .birthdate(LocalDate.of(2023, 1, 1))
          .addPreventionActsItem(
            new ImmunizationAct()
              .date(LocalDate.of(2023, 1, 1))
              .preventionMethodId("e05d945b-1e8c-4670-b2d0-281ebeea0a1c")
              .booster(false)
          )
      );

    try {
      var result = apiInstance.diagnosticForPatient(diagnosticRequest);
      System.out.println(
        "The immunization conclusion is \"" + result.getConclusion() + "\"."
      );
      for (var diseaseDiagnostic : result.getDiagnosticPerDisease()) {
        System.out.println(
          "The patient has status \"" +
          diseaseDiagnostic.getAdvice().getConclusion() +
          "\" for " +
          diseaseDiagnostic.getDisease().getName() +
          "."
        );
      }
    } catch (ApiException e) {
      System.err.println(
        "Exception when calling DefaultApi#diagnosticForPatient"
      );
      System.err.println("Status code: " + e.getCode());
      System.err.println("Reason: " + e.getResponseBody());
      System.err.println("Response headers: " + e.getResponseHeaders());
      e.printStackTrace();
    }
  }
}
